﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Tessa.Platform.Collections;
using Tessa.Platform.Data;
using Tessa.Roles.Acl;
using Tessa.Roles.NestedRoles;
using Tessa.Roles.Triggers;

namespace Tessa.Extensions.AclExamples.Server.Cards
{
    public sealed class AeAuthorAclGenerationRule : AclGenerationRuleBase
    {
        #region Nested Types

        private class InnerReader(
            AclGenerationRuleCardsResult cardsResult,
            IAclGenerationRule rule,
            IAclLockStrategy lockStrategy,
            IDbScope dbScope,
            INestedRoleManager nestedRoleManager)
            : AclGenerationRuleRolesReaderBase(cardsResult, rule, lockStrategy)
        {
            private readonly INestedRoleManager nestedRoleManager = NotNullOrThrow(nestedRoleManager);
            private readonly IDbScope dbScope = NotNullOrThrow(dbScope);

            private Dictionary<Guid, IList<Guid>> roleIDsByCardID;

            protected override async ValueTask<bool> PrepareRoleIDsAsync(CancellationToken cancellationToken = default)
            {
                if (this.CardsResult.CardIDs.Count == 1
                    && this.CardsResult.AdditionalData?.Count == 1
                    && this.CardsResult.AdditionalData[0].TryGetValue("AuthorID", out var authorID))
                {
                    var roleIDs = new List<Guid> { (Guid) authorID };
                    var nestedRoles = await this.nestedRoleManager.GetNestedRolesForCardAsync(roleIDs, this.CardID, cancellationToken);
                    foreach (var nestedRole in nestedRoles)
                    {
                        roleIDs.Add(nestedRole.ID);
                    }
                    this.RoleIDs = roleIDs;
                    return true;
                }

                if (await this.TryObtainLockForCardAsync(this.CardID, cancellationToken))
                {
                    if (this.roleIDsByCardID is null)
                    {
                        this.roleIDsByCardID = new Dictionary<Guid, IList<Guid>>();
                        await using var s = this.dbScope.Create();

                        var db = this.dbScope.Db;
                        var builder = this.dbScope.BuilderFactory;

                        int from = 0,
                            to = this.CardsResult.CardIDs.Count;

                        while (from < to)
                        {
                            db.SetCommand(
                                builder
                                    .Select().C("dci", "ID", "AuthorID")
                                    .From("DocumentCommonInfo", "dci").NoLock()
                                    .Where().C("ID").InArray(this.CardsResult.CardIDs.Skip(from).Take(Math.Min(1000, to - from)), "ids", out var dpIDs)
                                    .Build(), DataParameters.Get(dpIDs))
                                .LogCommand();

                            await using (var reader = await db.ExecuteReaderAsync(cancellationToken))
                            {
                                while (await reader.ReadAsync(cancellationToken))
                                {
                                    var currentCardID = reader.GetGuid(0);
                                    var currentAuthorID = reader.GetNullableGuid(1);
                                    this.roleIDsByCardID[currentCardID] = currentAuthorID is null ? Array.Empty<Guid>() : new List<Guid> { currentAuthorID.Value };
                                }
                            }

                            from += 1000;
                        }

                        var nestedRolesByCardID = await this.nestedRoleManager.GetNestedRolesForCardsAsync(this.roleIDsByCardID, cancellationToken);
                        foreach (var (cardID, nestedRoles) in nestedRolesByCardID)
                        {
                            if (nestedRoles.Count > 0)
                            {
                                this.roleIDsByCardID[cardID].AddRange(nestedRoles.Select(x => x.ID));
                            }
                        }
                    }

                    this.RoleIDs = this.roleIDsByCardID.TryGetValue(this.CardID, out var roleIDs) ? roleIDs.AsReadOnlyList() : [];

                    return true;
                }

                return false;
            }
        }

        #endregion

        #region Fields

        private readonly IDbScope dbScope;
        private readonly IAclLockStrategy lockStrategy;
        private readonly INestedRoleManager nestedRoleManager;

        #endregion

        #region Constructors

        public AeAuthorAclGenerationRule(
            IDbScope dbScope,
            IAclLockStrategy lockStrategy,
            INestedRoleManager nestedRoleManager)
            : base(AclExamplesConsts.AeAuthorRuleID, "Программный Автор")
        {
            ThrowIfNull(dbScope);
            ThrowIfNull(lockStrategy);
            ThrowIfNull(nestedRoleManager);

            this.dbScope = dbScope;
            this.lockStrategy = lockStrategy;
            this.nestedRoleManager = nestedRoleManager;
        }

        #endregion

        #region Base Overrides

        public override async ValueTask<AclGenerationRuleCardsResult> GetAllUpdateCardsAsync(CancellationToken cancellationToken = default)
        {
            await using var s = this.dbScope.Create();

            var db = this.dbScope.Db;
            var builder = this.dbScope.BuilderFactory;

            var cardIDs = await db.SetCommand(
                builder
                    .Select().C("ID")
                    .From("DocumentCommonInfo").NoLock()
                    .Build())
                .LogCommand()
                .ExecuteListAsync<Guid>(cancellationToken);


            return new AclGenerationRuleCardsResult
            {
                CardIDs = cardIDs,
                IgnoreRuleCheck = true,
            };
        }

        public override ValueTask<CheckTriggersResult> GetUpdateCardsByTriggersAsync(CheckTriggersRequest request, CancellationToken cancellationToken = default)
        {
            if (request.TriggerCard is { } card
                && card.Sections.TryGetValue("DocumentCommonInfo", out var dciSection)
                && dciSection.RawFields.TryGetValue("AuthorID", out var authorID))
            {
                return new ValueTask<CheckTriggersResult>(
                    new CheckTriggersResult
                    {
                        CardIDsWithData = new Dictionary<Guid, Dictionary<string, object>>
                        {
                            [card.ID] = new Dictionary<string, object> { ["AuthorID"] = authorID },
                        }
                    });
            }

            return new ValueTask<CheckTriggersResult>(CheckTriggersResult.GetEmpty());
        }

        public override IAclGenerationRuleRolesReader GetRolesReader(AclGenerationRuleCardsResult getCardsResult)
        {
            return new InnerReader(
                getCardsResult,
                this,
                this.lockStrategy,
                this.dbScope,
                this.nestedRoleManager);
        }

        #endregion
    }
}
