﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Tessa.Cards;
using Tessa.Cards.Extensions;
using Tessa.Extensions.Default.Shared.Settings;
using Tessa.Extensions.Default.Shared.Workflow.Wf;
using Tessa.Localization;
using Tessa.Platform;
using Tessa.Platform.Data;
using Tessa.Platform.Storage;

namespace Tessa.Extensions.WorkflowExamples.Server.Workflow
{
    /// <summary>
    /// Загрузка карточки с заполнением виртуальных секций в её заданиях.
    /// Метод выполняется при загрузке для всех карточек во всех режимах CardGetMethod, кроме Storage,
    /// но действительную работу выполняет только для заданий Workflow, у которых загружены секции.
    /// </summary>
    public sealed class WfTasksServerGetExtension :
        CardGetExtension
    {
        #region Constructors

        public WfTasksServerGetExtension(KrSettingsLazy settingsLazy)
        {
            this.settingsLazy = settingsLazy;
        }

        #endregion

        #region ChildResolutionInfo Private Class

        private sealed class ChildResolutionInfo
        {
            #region Constructors

            public ChildResolutionInfo(
                Guid rowID,
                string comment,
                string answer,
                DateTime created,
                DateTime planned,
                Guid performerID,
                string performerName,
                DateTime? inProgress,
                Guid? userID,
                string userName,
                DateTime? completed,
                Guid? optionID,
                string optionCaption)
            {
                this.RowID = rowID;
                this.Comment = comment;
                this.Answer = answer;
                this.Created = created;
                this.Planned = planned;
                this.PerformerID = performerID;
                this.PerformerName = performerName;
                this.InProgress = inProgress;
                this.UserID = userID;
                this.UserName = userName;
                this.Completed = completed;
                this.OptionID = optionID;
                this.OptionCaption = optionCaption;
            }

            #endregion

            #region Properties

            public Guid RowID { get; }

            public string Comment { get; }

            public string Answer { get; }

            public DateTime Created { get; }

            public DateTime Planned { get; }

            public Guid PerformerID { get; }

            public string PerformerName { get; }

            public DateTime? InProgress { get; }

            public Guid? UserID { get; }

            public string UserName { get; }

            public DateTime? Completed { get; }

            public Guid? OptionID { get; }

            public string OptionCaption { get; }

            #endregion
        }

        #endregion

        #region Fields

        private readonly KrSettingsLazy settingsLazy;

        #endregion

        #region Private Methods

        private static string TryGetTitleFromSection(CardTask task)
        {
            Card taskCard = task.TryGetCard();
            StringDictionaryStorage<CardSection> taskSections;
            return taskCard != null
                && (taskSections = taskCard.TryGetSections()) != null
                && taskSections.TryGetValue(WfHelper.CommonInfoSection, out CardSection commonInfoSection)
                    ? commonInfoSection.RawFields.TryGet<string>(WfHelper.CommonInfoKindCaptionField)
                    : null;
        }


        private static Task<string> TryGetTitleFromDbAsync(
            CardTask task,
            DbManager db,
            IQueryBuilderFactory builderFactory,
            CancellationToken cancellationToken = default)
        {
            return db
                .SetCommand(
                    builderFactory
                        .Select().C("KindCaption")
                        .From("TaskCommonInfo").NoLock()
                        .Where().C("ID").Equals().P("ID")
                        .Build(),
                    db.Parameter("ID", task.RowID))
                .LogCommand()
                .ExecuteAsync<string>(cancellationToken);
        }

        private static Task<string> TryGetTitleFromDbAsync(
            CardTaskHistoryItem task,
            DbManager db,
            IQueryBuilderFactory builderFactory,
            CancellationToken cancellationToken = default)
        {
            return db
                .SetCommand(
                    builderFactory
                        .Select().C("TypeCaption")
                        .From("TaskHistory").NoLock()
                        .Where().C("RowID").Equals().P("RowID")
                        .Build(),
                    db.Parameter("RowID", task.RowID))
                .LogCommand()
                .ExecuteAsync<string>(cancellationToken);
        }


        private static List<ChildResolutionInfo> TryGetChildrenInfo(CardTask task)
        {
            Card taskCard = task.TryGetCard();
            StringDictionaryStorage<CardSection> taskSections;
            ListStorage<CardRow> childrenRows;
            if (taskCard == null
                || (taskSections = taskCard.TryGetSections()) == null
                || !taskSections.TryGetValue(WfHelper.ResolutionChildrenSection, out CardSection childrenSection)
                || (childrenRows = childrenSection.TryGetRows()) == null
                || childrenRows.Count == 0)
            {
                return null;
            }

            var result = new List<ChildResolutionInfo>(childrenRows.Count);
            foreach (CardRow childrenRow in childrenRows)
            {
                result.Add(
                    new ChildResolutionInfo(
                        childrenRow.RowID,
                        childrenRow.Get<string>("Comment"),
                        childrenRow.Get<string>("Answer"),
                        childrenRow.Get<DateTime>("Created"),
                        childrenRow.Get<DateTime>("Planned"),
                        childrenRow.Get<Guid>("PerformerID"),
                        childrenRow.Get<string>("PerformerName"),
                        childrenRow.Get<DateTime?>("InProgress"),
                        childrenRow.Get<Guid?>("UserID"),
                        childrenRow.Get<string>("UserName"),
                        childrenRow.Get<DateTime?>("Completed"),
                        childrenRow.Get<Guid?>("OptionID"),
                        childrenRow.Get<string>("OptionCaption")));
            }

            return result;
        }


        private async ValueTask SetChildrenInfoToVirtualAsync(
            CardTask task,
            List<ChildResolutionInfo> infoList,
            CancellationToken cancellationToken = default)
        {
            if (infoList == null || infoList.Count == 0)
            {
                return;
            }

            CardSection virtualSection = task.Card.Sections.GetOrAdd(WfHelper.ResolutionChildrenVirtualSection);
            virtualSection.Type = CardSectionType.Table;
            virtualSection.TableType = CardTableType.Collection;

            ListStorage<CardRow> rows = virtualSection.Rows;
            rows.Clear();

            foreach (ChildResolutionInfo info in infoList.OrderBy(x => x.Created))
            {
                CardRow row = rows.Add();
                await this.StoreChildResolutionInfoAsync(info, row, cancellationToken);
                // по умолчанию уже установлено: row.State = CardRowState.None;
            }
        }


        private async ValueTask StoreChildResolutionInfoAsync(
            ChildResolutionInfo info,
            CardRow row,
            CancellationToken cancellationToken = default)
        {
            row.RowID = info.RowID;
            row["Comment"] = info.Comment;
            row["Answer"] = LocalizationManager.Format(info.Answer);
            row["Created"] = info.Created;
            row["Planned"] = info.Planned;
            row["PerformerID"] = info.PerformerID;
            row["PerformerName"] = info.PerformerName;
            row["InProgress"] = info.InProgress;
            row["UserID"] = info.UserID;
            row["UserName"] = info.UserName;
            row["Completed"] = info.Completed;
            row["OptionID"] = info.OptionID;
            row["OptionCaption"] = info.OptionCaption;
            row["ColumnComment"] = await this.GetColumnCommentAsync(info, cancellationToken);
            row["ColumnState"] = await this.GetColumnStateAsync(info, cancellationToken);
        }


        private async ValueTask<string> GetColumnCommentAsync(ChildResolutionInfo info, CancellationToken cancellationToken = default)
        {
            return info.Comment
                .NormalizeComment()
                .Limit((await this.settingsLazy.GetValueAsync(cancellationToken)).ChildResolutionColumnCommentMaxLength);
        }


        private async ValueTask<string> GetColumnStateAsync(ChildResolutionInfo info, CancellationToken cancellationToken = default)
        {
            return WfHelper.GetResolutionState(
                await this.settingsLazy.GetValueAsync(cancellationToken),
                info.PerformerName,
                info.UserID.HasValue ? info.UserName : null,
                info.OptionID);
        }

        #endregion

        #region Base Overrides

        public override async Task AfterRequest(ICardGetExtensionContext context)
        {
            Card card;
            if (!context.RequestIsSuccessful
                || context.CardType == null
                || context.CardType.Flags.HasNot(CardTypeFlags.AllowTasks)
                || context.Request.RestrictionFlags.Has(CardGetRestrictionFlags.RestrictTaskSections)
                || (card = context.Response.TryGetCard()) == null)
            {
                return;
            }

            await using (context.DbScope.Create())
            {
                ListStorage<CardTask> tasks = card.TryGetTasks();

                if (tasks != null && tasks.Count > 0)
                {
                    foreach (CardTask task in tasks)
                    {
                        if (TaskTypeIsResolution(task.TypeID))
                        {
                            if (task.IsLocked)
                            {
                                string title = await TryGetTitleFromDbAsync(
                                    task, context.DbScope.Db, context.DbScope.BuilderFactory, context.CancellationToken);

                                if (!string.IsNullOrEmpty(title))
                                {
                                    // если в строке NULL, то ExecuteScalar<string> вернёт пустую строку
                                    task.SetTitle(title);
                                }
                            }
                            else
                            {
                                string title = TryGetTitleFromSection(task);
                                if (title != null)
                                {
                                    task.SetTitle(title);
                                }

                                List<ChildResolutionInfo> infoList = TryGetChildrenInfo(task);
                                await this.SetChildrenInfoToVirtualAsync(task, infoList, context.CancellationToken);

                                Dictionary<string, object> fields = task.Card.Sections[WfHelper.ResolutionVirtualSection].RawFields;
                                fields[WfHelper.ResolutionVirtualPlannedField] = task.Planned;
                                fields[WfHelper.ResolutionVirtualDigestField] = await LocalizationManager.FormatAsync(task.Digest);
                            }
                        }
                    }
                }

                ListStorage<CardTaskHistoryItem> taskHistory = card.TryGetTaskHistory();

                if (taskHistory != null && taskHistory.Count > 0)
                {
                    foreach (CardTaskHistoryItem item in taskHistory)
                    {
                        if (TaskTypeIsResolution(item.TypeID))
                        {
                            // Если есть активное задание
                            var task = tasks?.FirstOrDefault(x => x.RowID == item.RowID);
                            if (task != null)
                            {
                                item.TypeCaption = TryGetTitleFromSection(task) ?? item.TypeCaption;
                            }
                            else
                            {
                                item.TypeCaption = await TryGetTitleFromDbAsync(
                                    item, context.DbScope.Db, context.DbScope.BuilderFactory, context.CancellationToken);
                            }
                        }
                    }
                }
            }
        }

        #endregion

        #region Task Types

        /// <summary>
        /// Task type identifier for "WfWeResolution": {77CC2212-067A-4190-987C-5F0EA0AE0725}.
        /// </summary>
        public static readonly Guid WfWeResolutionTypeID = new Guid(0x77cc2212, 0x067a, 0x4190, 0x98, 0x7c, 0x5f, 0x0e, 0xa0, 0xae, 0x07, 0x25);

        /// <summary>
        /// Task type name for "WfWeResolution".
        /// </summary>
        public const string WfWeResolutionTypeName = "WfWeResolution";

        /// <summary>
        /// Task type identifier for "WfWeResolutionProject": {946E6FC6-BE86-446D-B952-5C71EDB0A4EF}.
        /// </summary>
        public static readonly Guid WfWeResolutionProjectTypeID = new Guid(0x946e6fc6, 0xbe86, 0x446d, 0xb9, 0x52, 0x5c, 0x71, 0xed, 0xb0, 0xa4, 0xef);

        /// <summary>
        /// Task type name for "WfWeResolutionProject".
        /// </summary>
        public const string WfWeResolutionProjectTypeName = "WfWeResolutionProject";

        #endregion

        #region Static Methods

        /// <summary>
        /// Идентификаторы всех типов заданий, которые относятся к резолюциям Workflow.
        /// </summary>
        public static readonly Guid[] ResolutionTaskTypeIDList =
        {
            WfWeResolutionTypeID,
            WfWeResolutionProjectTypeID,
        };

        private static readonly HashSet<Guid> resolutionTaskTypeIDHashSet =
            new HashSet<Guid>(ResolutionTaskTypeIDList);

        public static bool TaskTypeIsResolution(Guid taskTypeID)
        {
            return resolutionTaskTypeIDHashSet.Contains(taskTypeID);
        }

        #endregion
    }
}
