﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Tessa.BusinessCalendar;
using Tessa.Cards;
using Tessa.Cards.ComponentModel;
using Tessa.Platform.Data;
using Tessa.Platform.Validation;
using Tessa.Roles;
using Tessa.Roles.ContextRoles;
using Tessa.Roles.Deputies;

namespace Tessa.Extensions.WorkflowExamples.Shared.Roles
{
    /// <summary>
    /// Выполняет агрегацию ролей, где результирующей ролью является роль задания.
    /// </summary>
    public class RolesFormationStrategy : IRolesFormationStrategy
    {
        #region Private Fields

        private readonly IRoleGetStrategy roleGetStrategy;
        private readonly IDbScope dbScope;
        private readonly IBusinessCalendarService businessCalendarService;
        private readonly IContextRoleManager contextRoleManager;
        private readonly IDeputiesManagementStrategy deputiesManagementStrategy;
        private readonly ITransactionScope transactionScope;
        private readonly ICardLockingStrategy cardLockingStrategy;

        #endregion

        #region Constructor

        public RolesFormationStrategy(
            IRoleGetStrategy roleGetStrategy,
            IDbScope dbScope,
            IBusinessCalendarService businessCalendarService,
            IContextRoleManager contextRoleManager,
            IDeputiesManagementStrategy deputiesManagementStrategy,
            ITransactionScope transactionScope,
            ICardLockingStrategy cardLockingStrategy)
        {
            this.roleGetStrategy = NotNullOrThrow(roleGetStrategy);
            this.dbScope = NotNullOrThrow(dbScope);
            this.businessCalendarService = NotNullOrThrow(businessCalendarService);
            this.contextRoleManager = NotNullOrThrow(contextRoleManager);
            this.deputiesManagementStrategy = NotNullOrThrow(deputiesManagementStrategy);
            this.transactionScope = NotNullOrThrow(transactionScope);
            this.cardLockingStrategy = NotNullOrThrow(cardLockingStrategy);
        }

        #endregion

        #region IRolesFormationStrategy Implementation

        /// <inheritdoc/>
        public async ValueTask<(Guid ID, string Name)> GetRolesFormationAsync(
            Guid cardID,
            Dictionary<Guid, string> roles,
            CancellationToken cancellationToken = default)
        {
            TaskRole taskRole = null;
            if (roles.Count == 1)
            {
                (var id, var name) = roles.ElementAt(0);
                return (id, name);
            }

            if (roles.Count >= 2)
            {
                var roleUsers = new List<RoleUser>();
                foreach (var role in roles)
                {
                    var typeRole = await this.roleGetStrategy.GetRoleParamsAsync(role.Key, cancellationToken);
                    if (typeRole.Type != null)
                    {
                        roleUsers.AddRange(
                            (await this.roleGetStrategy.GetUsersAsync(role.Key, cancellationToken: cancellationToken))
                            .Select(x => new RoleUser(x.UserID, x.UserName)));
                    }
                }

                taskRole = RoleHelper.CreateTaskRole(roleUsers.Distinct().ToArray());
                taskRole.Name = "Исполнители шага";

                var cardTemporaryTaskRole = 
                    new CardTemporaryTaskRole(
                        this.dbScope.Db, 
                        this.dbScope.BuilderFactory, 
                        this.contextRoleManager, 
                        this.businessCalendarService,
                        this.deputiesManagementStrategy,
                        this.transactionScope,
                        this.cardLockingStrategy,
                        taskRole,
                        DateTime.MinValue,
                        null,
                        null,
                        Guid.Empty,
                        null,
                        null,
                        cardID);

                var validationResultBuilder = new ValidationResultBuilder();
                await cardTemporaryTaskRole.InsertAsync(validationResultBuilder, cancellationToken);

                if (!validationResultBuilder.IsSuccessful())
                {
                    throw new ValidationException(validationResultBuilder.Build());
                }

            }

            if (taskRole is null)
            {
                throw new InvalidOperationException($"Task role is null in {nameof(RolesFormationStrategy)}.{nameof(this.GetRolesFormationAsync)}.");
            }

            return (taskRole.ID, taskRole.Name);
        }

        #endregion
    }
}
