﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Tessa.UI.Cards;
using Tessa.UI.Cards.Controls.AutoComplete;
using Tessa.Views;
using Tessa.Views.Metadata;
using Tessa.Views.Metadata.Criteria;

namespace Tessa.Extensions.AclExamples.Client.UI
{
    public class MeetingProtocolUIExtension : CardUIExtension
    {
        #region RowInfo Private Class

        private sealed class RowInfo
        {
            #region Constructors

            public RowInfo(
                object rowValue,
                Dictionary<string, int> indicesByName)
            {
                this.row = (IList<object>)rowValue;
                this.indicesByName = indicesByName;
            }

            #endregion

            #region Fields

            private readonly IList<object> row;

            private readonly Dictionary<string, int> indicesByName;

            #endregion

            #region Methods

            public T Get<T>(string column)
            {
                int index = this.indicesByName[column];
                return (T)this.row[index];
            }


            public DateTime GetDateTime(string column)
            {
                int index = this.indicesByName[column];
                DateTime value = (DateTime)this.row[index];

                return value.Kind != DateTimeKind.Unspecified
                    ? value.ToUniversalTime()
                    : DateTime.SpecifyKind(value, DateTimeKind.Utc);
            }

            #endregion
        }

        #endregion

        #region Constants

        private const string CollegiatePartControlAlias = "CollegiatePartControl";
        private const string AuthorControlAlias = "AuthorControl";

        private const string СollegiatePartsViewAlias = "AclExamplesCollegiateParts";
        private const string AclUsersViewAlias = "AclUsers";

        #endregion

        #region Fields

        private readonly IViewService viewService;

        private ITessaView collegiatePartsView;
        private ITessaView aclUsersView;

        #endregion

        #region Constructors

        public MeetingProtocolUIExtension(IViewService viewService)
        {
            this.viewService = viewService;
        }

        #endregion

        #region Base Overrides

        public override async Task Initialized(ICardUIExtensionContext context)
        {
            AutoCompleteEntryViewModel collegiatePartControl;
            AutoCompleteEntryViewModel authorControl;
            if ((collegiatePartControl = context.Model.Controls.TryGet<AutoCompleteEntryViewModel>(CollegiatePartControlAlias)) == null ||
                (authorControl = context.Model.Controls.TryGet<AutoCompleteEntryViewModel>(AuthorControlAlias)) == null)
            {
                return;
            }

            var aclMeetingProtocolsSection = context.Card.Sections["AclMeetingProtocols"];

            collegiatePartControl.ValueSelected += async (sender, args) =>
            {
                var newCollegiatePart = args.Item.Reference;
                if (this.collegiatePartsView is null)
                {
                    if ((this.collegiatePartsView = 
                        await this.viewService
                            .GetByNameAsync(СollegiatePartsViewAlias, context.CancellationToken)
                            .ConfigureAwait(false)) is null)
                    {
                        // у пользователя нет доступа к представлению, или нет самого представления
                        return;
                    }
                }

                IViewMetadata collegiatePartsViewMetadata = await this.collegiatePartsView.GetMetadataAsync(context.CancellationToken).ConfigureAwait(false);
                ITessaViewRequest collegiatePartsViewRequest = new TessaViewRequest(collegiatePartsViewMetadata) { CalculateRowCounting = false };

                var parameters = new List<RequestParameter>();
                // состояние заданий = "Новое"
                IViewParameterMetadata statusParam = collegiatePartsViewMetadata.Parameters.FindByName("CollegiatePartIDParam");
                if (statusParam != null)
                {
                    parameters.Add(
                        new RequestParameterBuilder()
                            .WithMetadata(statusParam)
                            .AddCriteria(
                                new EqualsCriteriaOperator(),
                                "CollegiatePartIDParam",
                                newCollegiatePart)
                            .AsRequestParameter());
                }

                collegiatePartsViewRequest.Values = parameters;

                // получаем данные представления
                ITessaViewResult viewResult = 
                    await this.collegiatePartsView.GetDataAsync(collegiatePartsViewRequest, context.CancellationToken);

                IList<object> viewResultRows = viewResult?.Rows;
                if (viewResult is null || viewResultRows is null || viewResultRows.Count == 0)
                {
                    return;
                }

                string[] columns = (viewResult.Columns ?? Array.Empty<string>()).Cast<string>().ToArray();

                var indicesByName = new Dictionary<string, int>(columns.Length);
                for (int i = 0; i < columns.Length; i++)
                {
                    indicesByName[columns[i]] = i;
                }
                
                var row = new RowInfo(viewResultRows.First(), indicesByName);
                var organizationID = row.Get<Guid>("OrganizationID");
                var organizationName = row.Get<string>("OrganizationName");

                if ((Guid?) aclMeetingProtocolsSection.RawFields["CollegiatePartOrganizationID"] != organizationID)
                {
                    aclMeetingProtocolsSection.Fields["CollegiatePartOrganizationID"] = organizationID;
                    aclMeetingProtocolsSection.Fields["CollegiatePartOrganizationName"] = organizationName;
                }
            };

            authorControl.ValueSelected += async (sender, args) =>
            {
                var newAuthor = args.Item.Reference;

                if (this.aclUsersView is null)
                {
                    if ((this.aclUsersView =
                        await this.viewService
                            .GetByNameAsync(AclUsersViewAlias, context.CancellationToken)
                            .ConfigureAwait(false)) is null)
                    {
                        // у пользователя нет доступа к представлению, или нет самого представления
                        return;
                    }
                }

                IViewMetadata aclUsersViewViewMetadata = await this.aclUsersView.GetMetadataAsync(context.CancellationToken).ConfigureAwait(false);
                ITessaViewRequest aclUsersViewViewRequest = new TessaViewRequest(aclUsersViewViewMetadata) { CalculateRowCounting = false };

                var parameters = new List<RequestParameter>();
                // состояние заданий = "Новое"
                IViewParameterMetadata statusParam = aclUsersViewViewMetadata.Parameters.FindByName("UserID");
                if (statusParam != null)
                {
                    parameters.Add(
                        new RequestParameterBuilder()
                            .WithMetadata(statusParam)
                            .AddCriteria(
                                new EqualsCriteriaOperator(),
                                "UserID",
                                newAuthor)
                            .AsRequestParameter());
                }

                aclUsersViewViewRequest.Values = parameters;

                // получаем данные представления
                ITessaViewResult viewResult = 
                    await this.aclUsersView.GetDataAsync(aclUsersViewViewRequest, context.CancellationToken);

                IList<object> viewResultRows = viewResult?.Rows;
                if (viewResult is null || viewResultRows is null || viewResultRows.Count == 0)
                {
                    return;
                }

                string[] columns = (viewResult.Columns ?? Array.Empty<string>()).Cast<string>().ToArray();

                var indicesByName = new Dictionary<string, int>(columns.Length);
                for (int i = 0; i < columns.Length; i++)
                {
                    indicesByName[columns[i]] = i;
                }

                var row = new RowInfo(viewResultRows.First(), indicesByName);
                var departmentID = row.Get<Guid?>("DepartmentID");
                var departmentName = row.Get<string>("DepartmentName");
                var organizationID = row.Get<Guid?>("OrganizationID");
                var organizationName = row.Get<string>("OrganizationName");

                if ((Guid?)aclMeetingProtocolsSection.RawFields["AuthorDepartmentID"] != departmentID)
                {
                    aclMeetingProtocolsSection.Fields["AuthorDepartmentID"] = departmentID;
                    aclMeetingProtocolsSection.Fields["AuthorDepartmentName"] = departmentName;
                }

                if ((Guid?)aclMeetingProtocolsSection.RawFields["AuthorOrganizationID"] != organizationID)
                {
                    aclMeetingProtocolsSection.Fields["AuthorOrganizationID"] = organizationID;
                    aclMeetingProtocolsSection.Fields["AuthorOrganizationName"] = organizationName;
                }
            };
        }

        #endregion
    }
}