﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Tessa.BusinessCalendar;
using Tessa.Cards;
using Tessa.Extensions.Default.Shared;
using Tessa.Extensions.WorkflowExamples.Shared.Workflow;
using Tessa.Platform;
using Tessa.Platform.Collections;
using Tessa.Platform.Data;
using Tessa.Platform.Storage;
using Tessa.Platform.Validation;
using Tessa.Roles;
using Tessa.Workflow;
using Tessa.Workflow.Actions;
using Tessa.Workflow.Compilation;
using Tessa.Workflow.Helpful;
using Tessa.Workflow.Signals;
using Unity;

namespace Tessa.Extensions.WorkflowExamples.Server.Workflow
{
    public sealed class WfeLoadSetStateAction : WorkflowActionBase
    {
        #region Nested Types

        public class StepInfo
        {
            public int? Deadline { get; set; }
            public string TaskText { get; set; }
            public string TaskTextOnRevision { get; set; }
            public Guid ID { get; set; }
        }

        #endregion

        #region Consts

        public const string MainActionSection = "WfeLoanSetStateAction";
        public const string MainLoanSection = "WfeLACommonInfo";
        public const string StagesSection = "WfeStagesStatistics";
        public const string MainTaskSection = "WfeTasksCommonInfo";

        #endregion

        #region Fields

        private readonly ICardRepository cardRepository;
        private readonly IRoleRepository roleRepository;
        private readonly IBusinessCalendarService calendarService;

        #endregion

        #region Constructors

        public WfeLoadSetStateAction(
            [Dependency(CardRepositoryNames.ExtendedWithoutTransaction)]
            ICardRepository cardRepository,
            IRoleRepository roleRepository,
            IBusinessCalendarService calendarService)
            : base(WfeHelper.WfeLoanSetStateDescriptor)
        {
            this.cardRepository = cardRepository;
            this.roleRepository = roleRepository;
            this.calendarService = calendarService;
        }

        #endregion

        #region Methods Overrides

        protected override async Task ExecuteAsync(IWorkflowEngineContext context, IWorkflowEngineCompiled scriptObject)
        {
            if (context.Signal.Type == WorkflowSignalTypes.Default)
            {
                var card = await context.GetMainCardAsync(context.CancellationToken);

                int? stepID = await context.GetAsync<int?>(MainActionSection, "Step", "ID"),
                    stageID = await context.GetAsync<int?>(MainActionSection, "Stage", "ID"),
                    stateID = await context.GetAsync<int?>(MainActionSection, "State", "ID"),
                    prevStageID = (int?) card.Sections[MainLoanSection].Fields["ProcessStagesID"];

                string stageName = await context.GetAsync<string>(MainActionSection, "Stage", "Name"),
                    stepName = await context.GetAsync<string>(MainActionSection, "Step", "Name");

                if (!stageID.HasValue
                    || !prevStageID.HasValue
                    || stageID.Value != prevStageID.Value)
                {
                    if (prevStageID.HasValue)
                    {
                        await this.CompleteStageAsync(context, card, stageID, stageName);
                    }

                    if (stageID.HasValue)
                    {
                        await this.StartStageAsync(context, card, stageID.Value, stageName);
                    }
                }

                if (stageID.HasValue)
                {
                    context.ValidationResult.Add(
                        await this.SetStateIDAsync(
                            card.ID,
                            stateID.Value,
                            await context.GetAsync<string>(MainActionSection, "State", "Name"),
                            context.CancellationToken));

                    if (!context.ValidationResult.IsSuccessful())
                    {
                        return;
                    }
                }

                if (stepID.HasValue)
                {
                    await this.CalculateAndFillTaskInfoAsync(
                        context,
                        stepID.Value);
                }

                card.Sections[MainLoanSection].Fields["ProcessStepID"] = Int32Boxes.Box(stepID);
                card.Sections[MainLoanSection].Fields["ProcessStepName"] = stepName;
                card.Sections[MainLoanSection].Fields["ProcessStagesID"] = Int32Boxes.Box(stageID);
                card.Sections[MainLoanSection].Fields["ProcessStagesName"] = stageName;
            }
        }

        #endregion

        #region Private Method

        private async ValueTask CompleteStageAsync(
            IWorkflowEngineContext context,
            Card card,
            int? nextStageID,
            string nextStageName)
        {
            var now = DateTime.UtcNow;
            var stagesSection = card.Sections[StagesSection];
            CardRow row;
            var rowID = context.ProcessInstance.Hash.TryGet<Guid?>("StageRowID");
            if (rowID.HasValue
                && (row = stagesSection.Rows.FirstOrDefault(x => x.RowID == rowID.Value)) != null)
            {
                var startDate = row.Get<DateTime>("DateStart");

                row.Fields["DateFinish"] = now;
                row.Fields["NextStageID"] = Int32Boxes.Box(nextStageID);
                row.Fields["NextStageName"] = nextStageName;
                row.Fields["LabourIntensity"] = await this.calendarService.GetDateDiffAsync(startDate, now, cancellationToken: context.CancellationToken) * 0.25;
                row.State = CardRowState.Modified;

                var completeTask = context.StoreCard.Tasks.FirstOrDefault(x => x.Action == CardTaskAction.Complete);
                if (completeTask != null)
                {
                    var taskSection = completeTask.Card.Sections[MainTaskSection];
                    row.Fields["CauseReturnID"] = taskSection.Fields.TryGet<Guid?>("ReasonForReturnID");
                    row.Fields["CauseReturnName"] = taskSection.Fields.TryGet<string>("ReasonForReturnName");
                }
            }
        }

        private async Task StartStageAsync(
            IWorkflowEngineContext context,
            Card card,
            int stageID,
            string stageName)
        {
            var now = DateTime.UtcNow;
            var planned = await this.calendarService.AddWorkingQuantsToDateAsync(
                now,
                await GetStageDeadlineAsync(context.DbScope, stageID, context.CancellationToken) * 4,
                cancellationToken: context.CancellationToken);

            card.Sections[MainLoanSection].Fields["StageDataSart"] = now;
            card.Sections[MainLoanSection].Fields["StagePlaneDataFinish"] = planned;

            var stagesSection = card.Sections[StagesSection];
            var order = stagesSection.Rows.Count;

            var newRow = stagesSection.Rows.Add();
            newRow.RowID = Guid.NewGuid();
            newRow.State = CardRowState.Inserted;
            newRow.Fields["Iterator"] = Int32Boxes.Box(order);
            newRow.Fields["DateStart"] = now;
            newRow.Fields["CurrentStageID"] = Int32Boxes.Box(stageID);
            newRow.Fields["CurrentStageName"] = stageName;
            newRow.Fields["PlaneDataFinish"] = planned;
            newRow.Fields["LabourIntensity"] = DoubleBoxes.Zero;

            context.ProcessInstance.Hash["StageRowID"] = newRow.RowID;
        }

        private static async Task<int> GetStageDeadlineAsync(IDbScope dbScope, int stageID, CancellationToken cancellationToken = default)
        {
            var db = dbScope.Db;
            return await db.SetCommand(
                dbScope
                    .BuilderFactory
                    .Select()
                        .Top(1)
                        .C("Deadline")
                    .From("WfeSettingPipelineLoanStage").NoLock()
                    .Where().C("StageID").Equals().P("StageID")
                    .Limit(1)
                    .Build(),
                    db.Parameter("StageID", stageID))
                .LogCommand()
                .ExecuteAsync<int?>(cancellationToken) ?? 0;
        }

        private async Task CalculateAndFillTaskInfoAsync(IWorkflowEngineContext context, int stepID)
        {
            var db = context.DbScope.Db;

            var taskID = context.Signal.Hash.TryGet<ICollection<object>>("TaskIDs", EmptyHolder<object>.Collection).FirstOrDefault() as Guid?;
            var isRevoke = context.Signal.Hash.TryGet<bool>("IsRevoke");

            var stepInfo = await db.SetCommand(
                    context.DbScope.BuilderFactory
                        .Select()
                            .C("ID")
                            .C("TaskText")
                            .C("TaskTextOnRevision")
                            .C("Deadline")
                        .From("WfeStepConfiguration").NoLock()
                        .Where()
                            .C("StepIDID").Equals().P("StepID")
                            .And()
                            .C("SegmentID").Equals().P("SegmentID")
                        .Build(),
                    db.Parameter("StepID", stepID),
                    db.Parameter("SegmentID", context.ProcessInstance.Hash.TryGet<int>("Segment")))
                .LogCommand()
                .ExecuteAsync<StepInfo>(context.CancellationToken);

            if (stepInfo is null)
            {
                context.ValidationResult.AddError(
                    this,
                    "Не заданы настройки шага.");
                return;
            }

            context.NodeInstance.Hash["Digest"] = string.Format(
                "{0}\n\n{1}: {2}",
                isRevoke ? stepInfo.TaskTextOnRevision : stepInfo.TaskText,
                context.Session.User.Name,
                taskID.HasValue
                    ? (await context.GetTaskAsync(taskID.Value, context.CancellationToken))
                    ?.Card.Sections["WfeTasksCommonInfo"].RawFields.TryGet<string>("Comment") ?? string.Empty
                    : string.Empty);

            context.NodeInstance.Hash["Period"] = stepInfo.Deadline / 8.0;

            var roles = new Dictionary<Guid, string>();

            db.SetCommand(
                    context.DbScope.BuilderFactory
                        .Select()
                            .C("RoleID")
                            .C("RoleName")
                        .From("WfeListRoles").NoLock()
                            .Where().C("ID").Equals().P("ID")
                        .Build(),
                    db.Parameter("ID", stepInfo.ID))
                .LogCommand();

            await using (var reader = await db.ExecuteReaderAsync(context.CancellationToken))
            {
                while (await reader.ReadAsync(context.CancellationToken))
                {
                    roles[reader.GetValue<Guid>(0)] = reader.GetValue<string>(1);
                }
            }

            if (roles.Count == 0)
            {
                context.ValidationResult.AddError(
                    this,
                    "Не заданы исполнители шага.");
                return;
            }

            (var taskRoleID, var taskRoleName) = await WfeWorkflowHelper.GetRolesFormationAsync(
                roles, this.roleRepository, context.ProcessInstance.CardID, context.CancellationToken);

            WorkflowEngineHelper.Set(context.NodeInstance.Hash, taskRoleID, "Role", "ID");
            WorkflowEngineHelper.Set(context.NodeInstance.Hash, taskRoleName, "Role", "Name");
        }

        private async Task<ValidationResult> SetStateIDAsync(Guid mainCardID, int stateID, string stateName, CancellationToken cancellationToken = default)
        {
            var getRequest = new CardGetRequest
            {
                CardID = mainCardID,
                CardTypeID = DefaultCardTypes.KrSatelliteTypeID,
                RestrictionFlags = CardGetRestrictionValues.Satellite,
            };
            getRequest.SetForbidStoringHistory(true);

            var response = await this.cardRepository.GetAsync(getRequest, cancellationToken);

            if (!response.ValidationResult.IsSuccessful())
            {
                return response.ValidationResult.Build();
            }

            var sCard = response.Card;
            sCard.Sections["KrApprovalCommonInfo"].Fields["StateID"] = Int32Boxes.Box(stateID);
            sCard.Sections["KrApprovalCommonInfo"].Fields["StateName"] = stateName;
            sCard.Sections["KrApprovalCommonInfo"].Fields["StateChangedDateTimeUTC"] = DateTime.UtcNow;

            var storeResponse = await this.cardRepository.StoreAsync(new CardStoreRequest { Card = sCard }, cancellationToken);
            return storeResponse.ValidationResult.Build();
        }

        #endregion
    }
}
