﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Tessa.BusinessCalendar;
using Tessa.Cards;
using Tessa.Platform.Data;
using Tessa.Platform.Storage;
using Tessa.Platform.Validation;
using Tessa.Properties.Resharper;
using Tessa.Roles;
using Tessa.Workflow;
using Unity;

namespace Tessa.Extensions.WorkflowExamples.Shared.Workflow
{
    public static class WfeWorkflowHelper
    {
        [UsedImplicitly]
        public static async Task<Guid> StartExpertiseAsync(
            IWorkflowEngineContext context,
            Guid taskID,
            Guid departmentID,
            string departmentName,
            int deadline)
        {
            var expertiseType = context.Signal.Hash.TryGet<int>("ExpertiseType");
            string expertiseName;
            switch(expertiseType)
            {
                case 0:
                    expertiseName = "Экспертиза";
                    break;
                case 1:
                    expertiseName = "Консультация";
                    break;
                default:
                    context.ValidationResult.AddError(nameof(WfeWorkflowHelper), $"Передан не правильный тип экспертизы {expertiseType}");
                    return Guid.Empty;
            }

            var card = await context.GetMainCardAsync(context.CancellationToken);
            var now = DateTime.UtcNow;
            var expertiseSection = card.Sections.GetOrAddTable("WfeExpertiseStatistics");

            if (expertiseSection.Rows.Any(x =>
                !x.Get<DateTime?>("DateFinish").HasValue
                && x.Get<Guid>("DepartmentsID") == departmentID
                && x.Get<Guid>("ParentRowID") == taskID))
            {
                context.ValidationResult.AddError(
                    nameof(WfeWorkflowHelper),
                    $"Для экспертного подразделения '{departmentName}' уже запрошена экспертиза/консультация");
                return Guid.Empty;
            }

            var newRow = expertiseSection.Rows.Add();
            newRow.RowID = Guid.NewGuid();
            newRow.State = CardRowState.Inserted;
            newRow.Fields["DateStart"] = now;
            newRow.Fields["ExpertiseTypeID"] = expertiseType;
            newRow.Fields["ExpertiseTypeName"] = expertiseName;
            newRow.Fields["DepartmentsID"] = departmentID;
            newRow.Fields["DepartmentsName"] = departmentName;
            newRow.Fields["LabourIntensity"] = 0;
            newRow.Fields["PlaneDataFinish"] = await context.Container.Resolve<IBusinessCalendarService>()
                .AddWorkingQuantsToDateAsync(now, deadline * 4, cancellationToken: context.CancellationToken);
            newRow.Fields["ParentRowID"] = taskID;

            return newRow.RowID;
        }

        
        [UsedImplicitly]
        public static async Task FinishExpertiseAsync(
            IWorkflowEngineContext context,
            Guid expertiseRowID,
            string comment,
            Guid? resultID = null,
            string resultName = null)
        {
            comment ??= string.Empty;

            var card = await context.GetMainCardAsync(context.CancellationToken);
            var now = DateTime.UtcNow;
            var expertiseSection = card.Sections.GetOrAddTable("WfeExpertiseStatistics");
            var row = expertiseSection.Rows.First(x => x.RowID == expertiseRowID);

            row.State = CardRowState.Modified;
            row.Fields["DateFinish"] = now;
            row.Fields["EmployeeID"] = context.Session.User.ID;
            row.Fields["EmployeeName"] = context.Session.User.Name;
            row.Fields["ResultRowID"] = resultID;
            row.Fields["ResultResultName"] = resultName;
            row.Fields["Comment"] = comment;
            row.Fields["CommentLimit"] = comment.Substring(0, Math.Min(comment.Length, 10));
            row.Fields["LabourIntensity"] = await context.Container.Resolve<IBusinessCalendarService>()
                .GetDateDiffAsync(row.Get<DateTime>("DateStart"), now, cancellationToken: context.CancellationToken) * 0.25;
        }

        
        [UsedImplicitly]
        public static async Task<(string TaskText, string Name, int Deadline)> GetTaskInfoForDepartmentAsync(
            IDbScope dbScope,
            Guid departmentID,
            int segmentID,
            CancellationToken cancellationToken = default)
        {
            await using (dbScope.Create())
            {
                dbScope.Db.SetCommand(
                    @"select top 1
                        t.TaskText,
                        t2.Name,
                        t.Deadline
                    from dbo.WfeSegmentsForDepartment t with(nolock)
                    inner join dbo.WfeExpertDepartments t2 with(nolock) on t.ID = t2.ID
                      where t.ID = @DepartmentID and t.SegmentID = @segmentID",
                    dbScope.Db.Parameter("@DepartmentID", departmentID),
                    dbScope.Db.Parameter("@segmentID", segmentID))
                    .LogCommand();

                await using var reader = await dbScope.Db.ExecuteReaderAsync(cancellationToken);
                if (await reader.ReadAsync(cancellationToken))
                {
                    return (
                        reader.GetValue<string>(0),
                        reader.GetValue<string>(1),
                        reader.GetValue<int?>(2) ?? 0);
                }
            }
            
            return (null, null, 0);
        }

        
        [UsedImplicitly]
        public static async Task<Dictionary<Guid, string>> GetRoleByDepartmentAsync(
            IDbScope dbScope,
            Guid departmentID,
            int segmentID,
            CancellationToken cancellationToken = default)
        {
            await using (dbScope.Create())
            {
                dbScope.Db.SetCommand(
                    @"select em.RoleID, em.RoleName from dbo.WfeEmployeeDepartment em
                      inner join dbo.WfeSegmentsForDepartment s on
                      em.SegmentRowID = s.RowID
                      where s.SegmentID = @SegmentID and s.ID = @DepartmentID"
                    , dbScope.Db.Parameter("@DepartmentID", departmentID)
                    , dbScope.Db.Parameter("@SegmentID", segmentID))
                    .LogCommand();
                
                var res = new Dictionary<Guid, string>();
                await using (var reader = await dbScope.Db.ExecuteReaderAsync(cancellationToken))
                {
                    while (await reader.ReadAsync(cancellationToken))
                    {
                        res.Add(reader.GetValue<Guid>(0), reader.GetValue<string>(1));
                    }
                }
                
                return res;
            }
        }

        
        public static async ValueTask<(Guid ID, string Name)> GetRolesFormationAsync(
            Dictionary<Guid, string> roles,
            IRoleRepository roleRepository,
            Guid cardID,
            CancellationToken cancellationToken = default)
        {
            TaskRole taskRole = null;
            if (roles.Count == 1)
            {
                (Guid id, string name) = roles.ElementAt(0);
                return (id, name);
            }

            if (roles.Count >= 2)
            {
                List<RoleUser> roleUsers = new List<RoleUser>();
                foreach (KeyValuePair<Guid, string> role in roles)
                {
                    Role typeRole = await roleRepository.GetRoleAsync(role.Key, cancellationToken);
                    if (typeRole != null)
                    {
                        roleUsers.AddRange(
                            (await roleRepository.GetUsersAsync(typeRole, cancellationToken: cancellationToken))
                            .Select(x => new RoleUser(x.UserID, x.UserName)));
                    }
                }

                taskRole = RoleHelper.CreateTaskRole(roleUsers.Distinct().ToArray());
                taskRole.Name = "Исполнители шага";
                await roleRepository.InsertAsync(taskRole, cancellationToken);
            }

            if (taskRole is null)
            {
                throw new InvalidOperationException($"Task role is null in {nameof(WfeWorkflowHelper)}.{nameof(GetRolesFormationAsync)}");
            }

            return (taskRole.ID, taskRole.Name);
        }
    }
}
